'use client';

import { useState } from 'react';
import { motion } from 'framer-motion';
import { Award, CheckCircle2 } from 'lucide-react';
import { useLanguage } from '@/contexts/language-context';
import { PlaceHolderImages } from '@/lib/placeholder-images';
import Image from 'next/image';
import {
  Carousel,
  CarouselContent,
  CarouselItem,
  CarouselNext,
  CarouselPrevious,
} from '@/components/ui/carousel';
import Autoplay from 'embla-carousel-autoplay';
import { Card, CardContent } from '@/components/ui/card';

export default function IsoCertificates() {
  const { content } = useLanguage();
  const { isoCertificates } = content;

  const certificateImages = {
    'ISO 9001:2015': PlaceHolderImages.find((img) => img.id === 'iso-9001'),
    'ISO 14001:2015': PlaceHolderImages.find((img) => img.id === 'iso-14001'),
    'ISO 45001:2018': PlaceHolderImages.find((img) => img.id === 'iso-45001'),
  };

  return (
    <section
      id="iso-certificates"
      className="scroll-mt-20 py-20 px-4 relative overflow-hidden bg-gradient-to-br from-slate-50 via-white to-amber-50/20"
    >
      <div className="absolute top-1/2 left-1/2 -translate-x-1/2 -translate-y-1/2 w-[800px] h-[800px] bg-gradient-to-br from-amber-500/5 via-blue-500/5 to-transparent rounded-full blur-3xl"></div>

      <div className="relative max-w-7xl mx-auto z-10">
        <div className="text-center mb-16">
            <motion.div
                initial={{ opacity: 0, y: 20 }}
                whileInView={{ opacity: 1, y: 0 }}
                viewport={{ once: true }}
                transition={{ duration: 0.6 }}
            >
                <div className="inline-flex items-center gap-2 mb-4 px-4 py-2 rounded-full bg-gradient-to-r from-amber-500/10 via-blue-500/10 to-amber-500/10 border border-amber-500/20">
                    <Award className="w-4 h-4 text-amber-600" />
                    <span className="text-sm font-medium text-primary">{isoCertificates.badge}</span>
                </div>
                <h2 className="font-headline text-3xl font-bold bg-gradient-to-r from-amber-600 via-blue-600 to-amber-600 bg-clip-text text-transparent sm:text-4xl">
                    {isoCertificates.title}
                </h2>
                <p className="mt-6 text-lg text-slate-600 max-w-3xl mx-auto">
                    {isoCertificates.subtitle}
                </p>
            </motion.div>
        </div>

        <Carousel
          opts={{
            align: 'start',
            loop: true,
          }}
          plugins={[
            Autoplay({
              delay: 4000,
              stopOnInteraction: false,
            }),
          ]}
          className="w-full max-w-5xl mx-auto"
        >
          <CarouselContent className="-ml-4">
            {isoCertificates.certificates.map((cert, index) => {
              const certImage = certificateImages[cert.name as keyof typeof certificateImages];
              return (
                <CarouselItem key={index} className="pl-4 md:basis-1/2 lg:basis-1/3">
                  <motion.div
                    className="p-2 h-full"
                    initial={{ opacity: 0, y: 30 }}
                    whileInView={{ opacity: 1, y: 0 }}
                    viewport={{ once: true }}
                    transition={{ duration: 0.5, delay: index * 0.1 }}
                  >
                    <Card className="group h-full overflow-hidden rounded-2xl bg-white/60 backdrop-blur-lg border-2 border-white/50 shadow-xl hover:shadow-amber-500/20 transition-all duration-300 hover:-translate-y-2">
                      <CardContent className="p-4 flex flex-col items-center text-center">
                        {certImage && (
                          <div className="w-full aspect-[4/3] rounded-lg overflow-hidden mb-6 border-2 border-slate-200/50 group-hover:border-amber-400/50 transition-all shadow-inner bg-slate-100 p-2">
                            <Image
                              src={certImage.imageUrl}
                              alt={cert.name}
                              width={600}
                              height={400}
                              className="w-full h-full object-contain group-hover:scale-105 transition-transform duration-500"
                              data-ai-hint={certImage.imageHint}
                            />
                          </div>
                        )}
                        <div className="flex items-center gap-3 mb-3">
                            <div className="relative">
                                <CheckCircle2 className="w-6 h-6 text-amber-500" />
                                <div className="absolute inset-0 bg-amber-500/30 rounded-full blur-sm group-hover:animate-pulse"></div>
                            </div>
                            <h3 className="text-lg font-bold text-primary group-hover:text-amber-700 transition-colors">
                                {cert.name}
                            </h3>
                        </div>
                        <p className="text-sm text-muted-foreground">
                          {cert.description}
                        </p>
                      </CardContent>
                    </Card>
                  </motion.div>
                </CarouselItem>
              );
            })}
          </CarouselContent>
          <CarouselPrevious className="absolute left-[-1.5rem] top-1/2 -translate-y-1/2 bg-white/90 hover:bg-white shadow-xl border-2 border-primary/20 hover:border-accent transition-all duration-300" />
          <CarouselNext className="absolute right-[-1.5rem] top-1/2 -translate-y-1/2 bg-white/90 hover:bg-white shadow-xl border-2 border-primary/20 hover:border-accent transition-all duration-300" />
        </Carousel>
      </div>
    </section>
  );
}
